%18.086 Final Project
%By Joseph Kovac
%Primary Refs.:    Briggs et al. - "A Multigrid Tutorial"
%                  Dave Willis and Jaydeep Bardhan, in Prof. White's group
%                  Prof. Strang, General Advice
%Revised and submitted for OCW 9/12/05

%This file is an example of the interface of the solver.  This experiment
%makes a guess of 1 at all points for a Lapalace's equation problem
%with no sources or sinks and zero boundary conditions in 2D.  The correct
%solution is 0 at all points.  The experiment relaxes on the fine grid
%(grid 1) and makes multiple transfers to coarser grids.  Coarsening
%operations are made which transfer the data to a grid twice as coarse as
%the previous grid, i.e. grid 1 transfers to grid 2, grid 2 to grid 4, 4
%to 8, and 8 to 16.  The guess at the error is then refined by
%interpolating the data at the coarse grid to finer grids and ultimately
%correcting the original relaxation on the fine grid.  The program then
%plots the evolution of the error corrections computed on successively
%coarser grids, followed by the incorporations of those incremental
%corrections as the solver moves back to the fine grid in a V-cycle scheme.
%One can easily see the multigrid advantage by comparing the final output 
%to the output of simply allowing the relaxation on the fine grid to run for 
%as many steps as the multigrid scheme, stopping the solver there, and plotting 
%the result.



%Initialize user-defined variables
clear RelaxedData;
%--------------------
%Set up the simulation grid (must be square)
InteriorRows=127;%The number of interior grid rows
InteriorColumns=127;%The number of interior grid columns
w=2/3; %Set w in the Jacobi relaxation scheme

%Set up boundary conditions
%BCMatrix is a matrix of dimension (InteriorRows+2)x(InteriorColumns+2)
%with the boundary conditions of the exterior grid edges.  These are most
%easily set by drawing the appropriately sized image in a graphics program
%and saving the image as a bitmap.  Basic examples would be monochrome
%bitmaps with black representing 0 and white 1.  In the current
%implementation, intergrid transfers and the multigrid scheme do not
%support boundary conditions imposed on the interior of the grid, but only
%the boundary conditions on the grid edges.  Drawing boundaries on the
%interior of the grid will relax correctly, provided no grid transfers are
%made.


%BCMatrix=sparse(double(BCImage));
%Create a sparse matrix for BCs using the input monorchrome image BCImage 
%(user must import appropriately sized matrix into BCImage; matrix must be of size
%(InteriorRows+2)x(InteriorColumns+2). This option is commented out here as we 
%simply want zero BCs, and that's easily specified.

BCMatrix=zeros(129,129);

%End user defined entries
%-----------------------------------------
%Reodering of BC data for simulation
BCVector=BCMatrix(1:end);%Reorder the matrix for vector operations
BCPoints=find(BCVector~=0);%Find the nonzero boundary value entries

%Charge (i.e. interior grid source) implementation
charge=0*speye(InteriorRows+2,InteriorColumns+2);
%Here there are no grid sources.  Sources can be set to values using the same 
%scheme as for boundary conditions, using bitmap image imports.  The image must again 
%be of size (InteriorRows+2)x(InteriorColumns+2).  To use that approach,
%simply comment out the above charge=... line and allow charge to be an
%appropriately sized matrix which might have been imported from a
%monocrhome bitmap.

ChargeVector=charge(1:end);%Reorder charge (source) data
f=ChargeVector'+BCVector';%The right-hand side f has a component due to
%sources as well as boundary conditions

InitCond=ones(129,129);%Set the initial conditions of the guess.  Again, this could be
%imported from an image file
ResHistory=[]; %ResHistory stores the residual norm value after each step

%HERE
%----------------------------------------
%Relaxation scheme
%----------------------------------------
%function "Relax" performs a relaxation on the finest grid
%Inputs: 
%Param 1: InitCond: Initial grid "guess" values: matrix of dim (InteriorRows+2)x(InteriorColumns+2)
%Param 2: BCPoints: Vector of non-zero boundary condition points, determined automatically above
%Param 3: Number of steps to relax
%Param 4: f: the appropriately constructed right-hand side; incorporates
%sources and BCs
%Param 5: w: w in Jacobi, set earlier
%Param 6: Grid step size: The distance represented between two grid points
%Outputs:
%Param 1: uh: An (InteriorRows+2)x(InteriorColumns+2) sized matrix
%with the relaxation results after the prescribed number of steps
%Param 2: rh: An ordered vector of the residual values at each point
%Param 3: ResNorms: A vector of the norm of the residual after each step

%Relax on fine grid
[u1h r1h ResNorms]=Relax(InitCond,BCPoints,300,f,w,1/128);

figure;
subplot(5,2,1);
surf(u1h);
title('Initial Solution on Grid 1');

%Add the evolution of the residual to ResHistory
ResHistory=[ResHistory ResNorms];

u1h=u1h(1:end);%Reorder the matrix into a vector

%Restriction scheme
%Use the "DownsampleMatrix" function to restrict the data to the coarser
%grid
%Inputs:
%Param 1: The residual to restrict, obtained from the previous relaxation, 
%but wrapped back into a matrix using WrapVector.
%Outputs:
%Param 1: fh: The residual on the coarser grid

%Restrict to grid 2
f2h=DownsampleMatrix(WrapVector(r1h));

%Residual relaxation scheme
%Use the "RelaxResiduals" function to solve the residual problem
%Inputs:
%Param 1: The residual to relax
%Param 2: The number of steps to relax
%Param 3: w in Jacobi
%Param 4: The representative distance between grid points
%Param 5: The initial guess at the answer; 0 indicates all zeros, otherwise
%the parameter is the ordered vector of the guess

%Relax on grid 2
[u2h r2h ResNorms]=RelaxResiduals(f2h,300,w,1/64,0);
subplot(5,2,2);
surf(u2h);
title('Error Refinement After Relaxation on Grid 2');
%r2h contains the residual after the relaxation
%Add residual evolution to history
ResHistory=[ResHistory ResNorms];

%Restrict Again to grid 4
f4h=DownsampleMatrix(WrapVector(r2h));
%Relax on the coarser grid 4
[u4h r4h ResNorms]=RelaxResiduals(f4h,300,2/3,1/32,0);
ResHistory=[ResHistory ResNorms];
subplot(5,2,3);
surf(u4h);
title('Error Refinement After Relaxation on Grid 4');

%Restrict to grid 8
f8h=DownsampleMatrix(WrapVector(r4h));
%Relax on grid 8
[u8h r8h ResNorms]=RelaxResiduals(f8h,300,2/3,1/16,0);
ResHistory=[ResHistory ResNorms];
subplot(5,2,4);
surf(u8h);
title('Error Refinement After Relaxation on Grid 8');

%Restrict to grid 16
f16h=DownsampleMatrix(WrapVector(r8h));
%Relax on grid 16
[u16h r16h ResNorms]=RelaxResiduals(f16h,300,2/3,1/8,0);
ResHistory=[ResHistory ResNorms];
subplot(5,2,5);
surf(u16h);
title('Error Refinement After Relaxation on Grid 16');

%Begin the interpolation scheme
%Interpolate using UpsampleMatrix
%Inputs:
%Param 1: the matrix of data
%Outputs:
%Param 1: the upsampled and interpolated data

%Interpolate answer on grid 16 to grid 8
e8h=UpsampleMatrix(u16h);
%Refine data on grid 8
u8h=u8h(1:end)'+e8h(1:end)';
%Relax again on grid 8 to smooth
[u8h rb8 ResNorms]=RelaxResiduals(f8h,300,2/3,1/16,u8h(1:end));
ResHistory=[ResHistory ResNorms];
subplot(5,2,6);
surf(u8h);
title('Error on Grid 8 Corrected by Error on Grid 16');

%Interpolate answer on grid 8 to grid 4
e4h=UpsampleMatrix(u8h);
%Refine data on grid 4
u4h=u4h(1:end)'+e4h(1:end)';
%Relax again on grid 4 to smooth
[u4h rb4 ResNorms]=RelaxResiduals(f4h,300,2/3,1/32,u4h(1:end));
ResHistory=[ResHistory ResNorms];
subplot(5,2,7);
surf(u4h);
title('Error on Grid 4 Corrected by Error on Grid 8');

%Interpolate answer on grid 4 to grid 2
e2h=UpsampleMatrix(u4h); 
%Refine data on grid 2
u2h=u2h(1:end)'+e2h(1:end)';
%Relax again on grid 4 to smooth
[u2h rb2 ResNorms]=RelaxResiduals(f2h,300,2/3,1/64,u2h(1:end));
subplot(5,2,8);
surf(u2h);
title('Error on Grid 2 Corrected by Error on Grid 4');

%Record residual evolution
ResHistory=[ResHistory ResNorms];

%Interpolate answer on grid 2 to grid 1
e1h=UpsampleMatrix(WrapVector(u2h));
subplot(5,2,9);
surf(e1h);
title('Error on Grid 1 Corrected by Error on Grid 2');
%Correct the initial relaxation on the fine grid with the interpolated
%guess at the error
u1h=u1h'+e1h(1:end)';
%One last fine relaxation to smooth out inerpolation error
[ufh rfh ResNorms]=Relax(WrapVector(u1h),BCPoints,800,f,2/3,1/128);
subplot(5,2,10);
surf(ufh);
title('Corrected Guess on Grid 1');

ResHistory=[ResHistory ResNorms];
%RelaxedData contains the final guess at the answer
%ResHistory contains the evolution of the residual (not-normalized to grid
%size)
'done'