% Simulation of scattering from a rigid (hard) sphere

clear all

Ns=50; 				% Discretization of sphere
dtheta=pi/Ns;		% Theta integration increment
dphi=2*pi/Ns;		% Phi integration increment
Nh=50;				% Maximum harmonic
ka=10;				% Scale parameter
k=ka;					% Wavenumber in medium assuming a=1

fprintf('Calculating spherical geometry...\n'); 

% Generate sphere
[x,y,z]=sphere(Ns);

% Cartesian to polar
theta=atan2(sqrt(x.^2+y.^2),z);
phi=atan2(y,x);

% Derivative of plane wave incident field wrt r
Pi=i*k*cos(theta).*exp(i*ka.*cos(theta));

figure(1)
surf(x,y,z,real(Pi));
shading interp
axis equal
xlabel('x')
ylabel('y')
zlabel('z')
title(sprintf('r-Derivative of Plane Wave Incident Field at ka=%.1f (Real part)',ka)); 
colorbar

fprintf('Projecting incident field...\n')

for n=0:Nh
   % Compute spherical wave spectrum for the *derivative*
   % of the incident field wrt r by integrating Ymn with
   % the calculated derivative over a unit sphere.
   % We know m=0 because there is no azimuth dependence
   % cf Eq(6.93) on page 207.
   
   C(n+1)=sum(sum(Pi.*conj(Ymn(0,n,theta,phi)).*sin(theta)));
   C(n+1)=C(n+1)*dtheta*dphi/dhndx(n,ka);
end

figure(2)
plot((0:Nh),abs(C),'b')
title(sprintf('Spherical Wave Spectrum of Plane Wave r-Derivative (ka=%.1f)',ka))
xlabel('Spherical wave number (n/a)')
ylabel('P_n')

% Now calculate scattered field in x-z plane
theta=0:dtheta:2*pi;
Nt=length(theta);
r=1000.0; % far-field!

fprintf('Calculating scattered field...\n');

p=zeros(1,Nt);

% Again, we know that m=0
for n=0:Nh
   % Dividing by k integrates the derivative
   % field back into a pressure field - valid
   % only in the far field, which is OK here.
   p=p-dhndx(n,k*r)/k*C(n+1)*Ymn(0,n,theta,0);
end

p=p*r/exp(i*k*r);

figure(3)
polar(theta,abs(p),'b');
hold on

% Analytic result

p0=zeros(1,Nt);

for n=0:Nh
   Pn=legendre(n,cos(theta));
   p0=p0+(2*n+1)*djndx(n,ka)/dhndx(n,ka)*Pn(1,:);
end

p0=i*p0/k;

polar(theta,abs(p0),'r.');
legend('numerical','analytic');
title(sprintf('Scattered Field - Rigid Sphere: ka=%.1f',ka));
axis equal
hold off

if max(abs(p)) > 2
   axis([-2 2 -2 2])
end







