% Simulation of scattering from a pressure-release (soft) sphere

clear all

Ns=50; 				% Discretization of sphere
dtheta=pi/Ns;		% Theta integration increment
dphi=2*pi/Ns;		% Phi integration increment
Nh=50;				% Maximum harmonic
c=1520;				% Speed of sound
ka=10;				% Scale parameter
P0=1;					% Incident pressure
k=ka;					% Medium wavenumber assuming a=1

fprintf('Calculating spherical geometry...\n'); 

% Generate sphere
[x,y,z]=sphere(Ns);

% Cartesian to polar
theta=atan2(sqrt(x.^2+y.^2),z);
phi=atan2(y,x);

% Plane wave incident field
Pi=P0*exp(i*k*z);

figure(1)
surf(x,y,z,real(Pi));
shading interp
axis equal
xlabel('x')
ylabel('y')
zlabel('z')
title(sprintf('Plane wave incident field for ka = %.1f (real part)',ka)); 
colorbar

fprintf('Projecting incident field...\n')

for n=0:Nh
   % Compute spherical wave spectrum for incident field
	% by integrating Ymn with incident field over sphere
   % We know m=0 because there is no azimuth dependence
   % cf Eq(6.95) on page 207.
   
   C(n+1)=sum(sum(Pi.*conj(Ymn(0,n,theta,phi)).*sin(theta)));
   C(n+1)=C(n+1)*dtheta*dphi/hn(n,ka);
   
   % Compute analytic spherical wave spectrum
   % cf Eq(6.174) on page 227.
   
   A(n+1)=i^n*(2*n+1)*jn(n,ka)/hn(n,ka)/sqrt((2*n+1)/4/pi);
end

figure(2)
plot((0:Nh),abs(C),'b')
hold on
plot((0:Nh),abs(A),'r*')
hold off
title(sprintf('Spherical Wave Spectrum of Plane Wave (ka = %.1f)',ka))
xlabel('Spherical wave number (n/a)')
ylabel('P_n')
legend('numerical','analytic');

% Now calculate scattered field in x-z plane
theta=0:dtheta:2*pi;
Nt=length(theta);
r=1000.0; % far-field !

fprintf('Calculating scattered field...\n');

p=zeros(1,Nt);

% Again, we know that m=0
for n=0:Nh
   p=p-hn(n,k*r)*C(n+1)*Ymn(0,n,theta,0);
end

p=p*r/exp(i*k*r);

figure(3)
polar(theta,abs(p),'b');
hold on

% Analytic result
p=zeros(1,Nt);

for n=0:Nh
   Pn=legendre(n,cos(theta));
   p=p+(2*n+1)*jn(n,ka)/hn(n,ka)*Pn(1,:);
end

p=p/ka;

polar(theta,abs(p),'r.');
legend('numerical','analytic');
title(sprintf('Scattered Field - Pressure Release Sphere: ka=%.1f',ka));
hold off

if max(abs(p)) > 2
   axis([-2 2 -2 2])
end







